using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;

using gov.va.med.vbecs.Common;
using Sprocs = gov.va.med.vbecs.Common.VbecsStoredProcs;
using UserTable = gov.va.med.vbecs.Common.VbecsTables.VbecsUser;
using VbecsUserDivisionRoleTable = gov.va.med.vbecs.Common.VbecsTables.VbecsUserDivisionRole;

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Mike Smith</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/17/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// Data access component providing access VBECS to VBECS user information
	/// and user management functionality.
	///</summary>

	#endregion

	public class VbecsUser
	{
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="312"> 
		///		<ExpectedInput>Valid user ID</ExpectedInput>
		///		<ExpectedOutput>DataRow with all required columns.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2458"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads user information for a VBECS user with a given ID (NT login ID).
		/// Changed TO DataTable to handle Default User assignment 4/20/2005 CR717
		/// </summary>
		/// <param name="vbecsUserID">User NT login ID.</param>
		/// <param name="isDefaultUser">When true, the user is VBECS, a default, and sproc will behave differently</param>
		/// <returns>DataRow containing information about a VBECS user.</returns>
		public static DataTable GetRequireVbecsUser( string vbecsUserID, bool isDefaultUser )
		{
			if( vbecsUserID == null )
				throw( new ArgumentNullException( "vbecsUserID" ) );

			
			return Common.StoredProcedure.GetDataTable(
				Sprocs.GetVbecsUser.StoredProcName, 
				GetVbecsUserIdSqlParameter( Sprocs.GetVbecsUser.vbecsuserid, vbecsUserID),
				Common.StoredProcedure.MakeParameter(Sprocs.GetVbecsUser.searchallusers, isDefaultUser)
			);
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6904"> 
		///		<ExpectedInput>Valid user ID.</ExpectedInput>
		///		<ExpectedOutput>DataRow with all required columns.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6906"> 
		///		<ExpectedInput>Nonexistent user ID.</ExpectedInput>
		///		<ExpectedOutput>Null.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6905"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks if the user with specified NT login ID exists in the VBECS database
		/// and returns <see cref="DataRow"/> containing user details if the user is found, or null if not. 
		/// </summary>
		/// <param name="vbecsUserID">User NT login ID to look for.</param>
		/// <returns><see cref="DataRow"/> if user is found. Null if user is not found.</returns>
		public static DataRow GetCheckForVbecsUser( string vbecsUserID )
		{
			if( vbecsUserID == null )
				throw( new ArgumentNullException( "vbecsUserID" ) );

			DataTable _dt = Common.StoredProcedure.GetDataTable( 
				Sprocs.GetVbecsUser.StoredProcName, 
				GetVbecsUserIdSqlParameter( Sprocs.GetVbecsUser.vbecsuserid, vbecsUserID ) );

			if( _dt.Rows.Count == 0 )
				return null;

			if( _dt.Rows.Count > 1 )
				throw( new DataAccessLayerException( StrRes.SysErrMsg.Common.InsufficientNumberOfDataRowsFoundWhileExpectingSingleRow( Sprocs.GetVbecsUser.StoredProcName, _dt.Rows.Count ).ResString ) );

			return _dt.Rows[0];
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5007"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-empty list of users.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5008"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves all user records for all divisions from the database (including inactive).
		/// </summary>
		/// <returns><see cref="DataTable"/> containing data for all users in all divisions.</returns>
		public static DataTable GetAllUsers()
		{
			return Common.StoredProcedure.GetDataTable( Sprocs.GetAllVbecsUsers.StoredProcName );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8055"> 
		///		<ExpectedInput>UserData</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8056"> 
		///		<ExpectedInput>Invalid UserData</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PersistUsers
		/// </summary>
		/// <param name="usersCoreData"></param>
		/// <param name="usersDivisionRoles"></param>
		/// <param name="lastUpdateFunction"></param>
		public static void PersistUsers( DataTable usersCoreData, IList usersDivisionRoles, UpdateFunction lastUpdateFunction )
		{
			ArrayList storedProcs = new ArrayList();
			ArrayList dataTables = new ArrayList();

			AddVbecsUsersListSaveActions( storedProcs, dataTables, usersCoreData, usersDivisionRoles, lastUpdateFunction );
			ExecuteVbecsUserSaveActions( storedProcs, dataTables );
		}

		private static void ExecuteVbecsUserSaveActions( IList storedProcs, IList dataTables )
		{
			if( storedProcs == null )
				throw( new ArgumentNullException( "storedProcs" ) );

			if( dataTables == null )
				throw( new ArgumentNullException( "dataTables" ) );

			if( storedProcs.Count == 0 )
				return;			

			try
			{
				new StoredProcedure().TransactionalGetValue( storedProcs, dataTables );
			}
			catch( UniqueDatabaseConstraintViolatedException xcp )
			{
				AddUniqueConstraintViolationInfo( xcp );
				throw;
			}
		}

		internal static void AddVbecsUsersListSaveActions( IList storedProcs, IList dataTables, DataTable usersCoreData, IList usersDivisionRoles, UpdateFunction lastUpdateFunction )
		{
			if( storedProcs == null )
				throw( new ArgumentNullException( "storedProcs" ) );

			if( dataTables == null )
				throw( new ArgumentNullException( "dataTables" ) );

			if( usersCoreData == null )
				throw( new ArgumentNullException( "usersCoreData" ) );

			if( usersDivisionRoles == null )
				throw( new ArgumentNullException( "usersDivisionRoles" ) );

			foreach( DataRow drUserCore in usersCoreData.Rows )
				AddVbecsUserCoreSaveActions( storedProcs, dataTables, drUserCore, lastUpdateFunction );

			foreach( DataTable dtUserRoles in usersDivisionRoles )
				AddVbecsUserDivisionRoleSaveActions( storedProcs, dataTables, dtUserRoles, lastUpdateFunction );
		}

		internal static void AddVbecsUserSaveActions( IList storedProcs, IList dataTables, DataRow userDataRow, DataTable userDivisionRoles, UpdateFunction lastUpdateFunction )
		{
			if( storedProcs == null )
				throw( new ArgumentNullException( "storedProcs" ) );

			if( dataTables == null )
				throw( new ArgumentNullException( "dataTables" ) );

			if( userDataRow != null )
				AddVbecsUserCoreSaveActions( storedProcs, dataTables, userDataRow, lastUpdateFunction );
			
			if( userDivisionRoles != null )
				AddVbecsUserDivisionRoleSaveActions( storedProcs, dataTables, userDivisionRoles, lastUpdateFunction );
		}

		private static void AddVbecsUserCoreSaveActions( IList storedProcs, IList dataTables, DataRow userDataRow, UpdateFunction lastUpdateFunction )
		{
			if( storedProcs == null )
				throw( new ArgumentNullException( "storedProcs" ) );

			if( dataTables == null )
				throw( new ArgumentNullException( "dataTables" ) );

			if( userDataRow == null )
				throw( new ArgumentNullException( "userDataRow" ) );

			DataTable dtCoreUpdateTable = GetUserTemplateDataTable();
			dtCoreUpdateTable.ImportRow( userDataRow );
			dtCoreUpdateTable.Rows[0][ UserTable.LastUpdateFunctionId ] = (int)lastUpdateFunction;
						
			storedProcs.Add( StNullConvert.IsNull( userDataRow[ UserTable.RowVersion ] ) ? Sprocs.InsertVbecsUser.StoredProcName : Sprocs.UpdateVbecsUser.StoredProcName );
			dataTables.Add( dtCoreUpdateTable );
		}

		private static void AddVbecsUserDivisionRoleSaveActions( IList storedProcs, IList dataTables, DataTable userDivisionRoles, UpdateFunction lastUpdateFunction )
		{
			if( storedProcs == null )
				throw( new ArgumentNullException( "storedProcs" ) );

			if( dataTables == null )
				throw( new ArgumentNullException( "dataTables" ) );

			if( userDivisionRoles == null )
				throw( new ArgumentNullException( "userDivisionRoles" ) );

			DataTable dtRolesToInsert = GetVbecsUserDivisionRoleSaveTemplateDataTable();
			DataTable dtRolesToUpdate = GetVbecsUserDivisionRoleSaveTemplateDataTable();

			foreach( DataRow dtRow in userDivisionRoles.Rows )
			{				
				dtRow[ VbecsTables.VbecsUserDivisionRole.LastUpdateFunctionId ] = (int)lastUpdateFunction;

				if( StNullConvert.IsNull( dtRow[ VbecsTables.VbecsUserDivisionRole.RowVersion ] ) )
					dtRolesToInsert.ImportRow( dtRow );
				else
					dtRolesToUpdate.ImportRow( dtRow );
			}

			if( dtRolesToInsert.Rows.Count > 0 )
			{				
				storedProcs.Add( Sprocs.InsertVbecsUserDivisionRole.StoredProcName );
				dataTables.Add( dtRolesToInsert );
			}

			if( dtRolesToUpdate.Rows.Count > 0 )
			{
				storedProcs.Add( Sprocs.UpdateVbecsUserDivisionRole.StoredProcName );
				dataTables.Add( dtRolesToUpdate );
			}
		}

		/// <summary>
		/// Adds unique constraint violation information to the 
		/// <see cref="UniqueDatabaseConstraintViolatedException"/> exception.
		/// </summary>
		/// <param name="xcp">Exception to extend constrain violation information for.</param>
		private static void AddUniqueConstraintViolationInfo( UniqueDatabaseConstraintViolatedException xcp )
		{
			if( xcp == null )
				throw( new ArgumentNullException( "xcp" ) );

			if( xcp.Message == null ) // Not the situation we're interested in (looking for specific violations).
				return;

			string _flattenedMessage = xcp.Message.ToLower();

			if( _flattenedMessage.IndexOf( DatabaseConstants.VbecsUserUniqueConstraints.PkVbecsUserId.ToLower() ) != -1 )
				xcp.ConflictingDbColumnName = VbecsTables.VbecsUser.VbecsUserId;
			else
			if( _flattenedMessage.IndexOf( DatabaseConstants.VbecsUserUniqueConstraints.UcVbecsUserUserDuz.ToLower() ) != -1 )
				xcp.ConflictingDbColumnName = VbecsTables.VbecsUser.UserDUZ;
			else
			if( _flattenedMessage.IndexOf( DatabaseConstants.VbecsUserUniqueConstraints.UcVbecsUserUserInitials.ToLower() ) != -1 )
				xcp.ConflictingDbColumnName = VbecsTables.VbecsUser.UserInitials;
			else
			if( _flattenedMessage.IndexOf( DatabaseConstants.VbecsUserUniqueConstraints.UcVbecsUserVistaUserName.ToLower() ) != -1 )
				xcp.ConflictingDbColumnName = VbecsTables.VbecsUser.VistaUserName;
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1299"> 
		///		<ExpectedInput>Valid VBECS user</ExpectedInput>
		///		<ExpectedOutput>Data table with the permissions for the user.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1974"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a permissions for a user with specified ID. 
		/// </summary>
		/// <param name="vbecsUserID">User ID</param>
		/// <returns><see cref="DataTable"/> with permissions for a given user.</returns>
		public static DataTable GetVbecsUserPermissions( string vbecsUserID )
		{ 
			if( vbecsUserID == null )
				throw( new ArgumentNullException( "vbecsUserID" ) );

			return Common.StoredProcedure.GetDataTable( 
				Sprocs.UserPermissions.StoredProcName, 
				GetVbecsUserIdSqlParameter( Sprocs.UserPermissions.vbecsuserid, vbecsUserID ) );
		}
		
		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6909"> 
		///		<ExpectedInput>Valid user ID.</ExpectedInput>
		///		<ExpectedOutput>Valid DataTable containing user divisions and roles.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6910"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns <see cref="DataTable"/> containing user division roles information for 
		/// a user with a given NT login ID. 
		/// </summary>
		/// <param name="vbecsUserID">VBECS user ID to retrieve divisions and roles for.</param>
		/// <returns>
		///		<see cref="DataTable"/> containing a list of divisions 
		///		and roles for the VBECS user with the specified ID.
		///	</returns>
		public static DataTable GetVbecsUserDivisionsAndRoles( string vbecsUserID )
		{
			if( vbecsUserID == null )
				throw( new ArgumentNullException( "vbecsUserID" ) );

			return Common.StoredProcedure.GetDataTable( 
				Sprocs.GetVbecsUserDivisionsAndRoles.StoredProcName, 
				GetVbecsUserIdSqlParameter( Sprocs.GetVbecsUserDivisionsAndRoles.vbecsuserid, vbecsUserID ) );
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/23/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="309"> 
		///		<ExpectedInput>Valid user role id.</ExpectedInput>
		///		<ExpectedOutput>Data table of all the roles with the same or less permission than the input role.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2461"> 
		///		<ExpectedInput>Invalid Id</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///	Gets a list of all the user roles at or under the user role ID passed in
		/// </summary>
		/// <param name="userRoleId">User role ID.</param>
		/// <returns><see cref="DataTable"/> containing list of user roles.</returns>
		public static DataTable GetUserRolesList( int userRoleId )
		{
			SqlParameter prm =	new SqlParameter( Sprocs.GetUserRolesList.userroleid, SqlDbType.Int );
			prm.Value = userRoleId;
	
			return Common.StoredProcedure.GetDataTable( Sprocs.GetUserRolesList.StoredProcName, prm );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5005"> 
		///		<ExpectedInput>Valid division code.</ExpectedInput>
		///		<ExpectedOutput>Non-empty table with user data.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5006"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get active users for a given division.
		/// </summary>
		/// <param name="divisionCode">Division code</param>
		/// <returns><see cref="DataTable"/> with active division users.</returns>
		public static DataTable GetActiveDivisionUsers( string divisionCode )
		{
			ValidateDivisionCode( divisionCode );

			SqlParameter prm = new SqlParameter( Sprocs.GetVbecsUsersActiveForDivision.divisioncode, SqlDbType.Char );
			prm.Value = divisionCode;
		
			return Common.StoredProcedure.GetDataTable( Sprocs.GetVbecsUsersActiveForDivision.StoredProcName, prm );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8057"> 
		///		<ExpectedInput>DivisionCode</ExpectedInput>
		///		<ExpectedOutput>DataTable of User data</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8058"> 
		///		<ExpectedInput>Invalid DivisionCode</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetVbecsUsersWithRoleInGivenDivision
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public static DataTable GetVbecsUsersWithRoleInGivenDivision( string divisionCode )
		{
			ValidateDivisionCode( divisionCode );

			SqlParameter prm = new SqlParameter( Sprocs.GetVbecsUsersWithRoleInGivenDivision.divisioncode, SqlDbType.Char );
			prm.Value = divisionCode;
		
			return Common.StoredProcedure.GetDataTable( Sprocs.GetVbecsUsersWithRoleInGivenDivision.StoredProcName, prm );
		}

		private static void ValidateDivisionCode( string divisionCode ) 
		{
			if( divisionCode == null || divisionCode == string.Empty)
				throw( new ArgumentException( "divisionCode" ) );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8059"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8060"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetVbecsUserDivisionRoleSaveTemplateDataTable
		/// </summary>
		/// <returns></returns>
		public static DataTable GetVbecsUserDivisionRoleSaveTemplateDataTable()
		{
			DataTable dt = new DataTable();

			dt.Columns.Add( VbecsUserDivisionRoleTable.VbecsUserId, typeof(string ) );
			dt.Columns.Add( VbecsUserDivisionRoleTable.DivisionCode, typeof(string) );
			dt.Columns.Add( VbecsUserDivisionRoleTable.UserRoleId, typeof(int) );
			dt.Columns.Add( VbecsUserDivisionRoleTable.RowVersion, typeof(byte[]) );
			dt.Columns.Add( VbecsUserDivisionRoleTable.RecordStatusCode, typeof(string) );
			dt.Columns.Add( VbecsUserDivisionRoleTable.LastUpdateFunctionId, typeof(int) );

			return dt;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8061"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8062"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates template <see cref="DataTable"/> used for interlayer 
		/// VBECS user data exchange.
		/// </summary>
		/// <returns>VBECS user template <see cref="DataTable"/>.</returns>
		public static DataTable GetUserTemplateDataTable()
		{
			DataTable dt = new DataTable();

			dt.Columns.Add( UserTable.VbecsUserId, typeof(string) );
			dt.Columns.Add( UserTable.VbecsUserGuid, typeof(Guid) );
			dt.Columns.Add( UserTable.UserDUZ, typeof(string) );
			dt.Columns.Add( UserTable.UserName, typeof(string) );			
			dt.Columns.Add( UserTable.UserInitials, typeof(string) );
			dt.Columns.Add( UserTable.VistaUserName, typeof(string) );
			dt.Columns.Add( UserTable.UserEmail, typeof(string) );
			dt.Columns.Add( UserTable.RecordStatusCode, typeof(string) );
			dt.Columns.Add( UserTable.RowVersion, typeof(byte[]) );
			dt.Columns.Add( UserTable.LastUpdateFunctionId, typeof(int) );

			return dt;
		}
		
		/// <summary>
		/// Creates <see cref="SqlParameter"/> for VBECS user ID (NT login ID). 
		/// </summary>
		/// <param name="paramName">User ID parameter name (normally, vbecsuserid)</param>
		/// <param name="vbecsUserId">VBECS user ID to set parameter value to.</param>
		/// <returns><see cref="SqlParameter"/> with a given name and a given user ID value.</returns>
		internal static SqlParameter GetVbecsUserIdSqlParameter( string paramName, string vbecsUserId )
		{
			if( paramName == null )
				throw( new ArgumentNullException( "paramName" ) );

			if( vbecsUserId == null )
				throw( new ArgumentNullException( "vbecsUserId" ) );

			SqlParameter _prm = new SqlParameter( paramName, System.Data.SqlDbType.VarChar );
			_prm.Value = vbecsUserId;

			return _prm;
		}
	}
}